module.exports = {
    name: "Request API (AXIOS)",

    description: "Allows you to request an API using Axios. Finally working without any issues, i hope.",

    category: "Internet Stuff",

    inputs: [
        {
            "id": "action",
            "name": "Action",
            "description": "Acceptable Types: Action\n\nDescription: Executes this block.",
            "types": ["action"]
        },
        {
            "id": "url",
            "name": "API URL",
            "description": "Acceptable Types: Text, Unspecified\n\nDescription: The URL of API.",
            "types": ["text", "unspecified"],
            "required": true
        },
        {
            "id": "header",
            "name": "Header",
            "description": "Acceptable Types: Object, Unspecified\n\nDescription: The headers to send to API. (OPTIONAL)\n\nNEW METHOD: With the new Multi Input you can input text here!\nExample:\n  Header#1: x-api-key: XXXX-XXX-XXXXXXX\n  Header#2: Content-Type: application/json\netc....\n\n\nYou can still connect a Single Object into here and it would merge them.",
            "types": ["object", "text", "unspecified"],
            "multiInput": true
        },
        {
            "id": "body",
            "name": "Body",
            "description": "Acceptable Types: Text, Object, Unspecified\n\nDescription: The body to send to API. (OPTIONAL)\n\nNEW METHOD: With the new Multi Input you can input text here!\nExample:\n  Body#1: \"key1\": \"value\"\n  Body#2: \"key2\": \"value\"\netc....\n\n\nYou can still connect a Single Object into here and it would merge them.",
            "types": ["object", "text", "unspecified"],
            "multiInput": true
        }
    ],

    options: [
        {
            "id": "method_type",
            "name": "Method Type",
            "description": "Description: The type of method for API request.",
            "type": "SELECT",
            "options": {
                "get": "GET",
                "post": "POST",
                "put": "PUT",
                "delete": "DELETE",
                "patch": "PATCH"
            }
        },
        {
            "id": "return_type",
            "name": "Return Type",
            "description": "Description: The type of data to obtain from API request.",
            "type": "SELECT",
            "options": {
                "json": "Object/Text",
				"buffer": "Buffer",
                "attachment": "Attachment"
            }
        },
        {
            id: "timeout",
            name: "Timeout",
            description: "Description: The time in milliseconds before the request times out.\nDefault: 0 (No timeout)",
            type: "NUMBER",
            defaultValue: 0
        }
    ],

    outputs: [
        {
            "id": "action",
            "name": "Action",
            "description": "Type: Action\n\nDescription: Executes the following blocks when this block finishes its task.",
            "types": ["action"]
        },
        {
            "id": "status",
            "name": "Status Code",
            "description": "Type: Number\n\nDescription: The status code of the API request.",
            "types": ["number"]
        },
        {
            "id": "data",
            "name": "API Data",
            "description": "Type: Object\n\nDescription: The API data obtained if possible.",
            "types": ["object", "unspecified"]
        },
        {
            "id": "headers",
            "name": "Response Headers",
            "description": "Type: Object\n\nDescription: The headers of the API request.",
            "types": ["object"]
        }
    ],

    async code(cache) {
        const { AttachmentBuilder } = require("discord.js");
        const url = this.GetInputValue("url", cache);

        let headers = this.GetInputValue("header", cache);
        try {
            if(headers && headers.length >= 1 && typeof headers[0] === "string") {
                temp = {};
                headers.forEach(h => {
                    const parts = h.split(h.includes(": ") ? ": " : ":");
                    temp[parts[0]] = parts[1];
                });
                headers = temp;
            } else if(headers && headers.length >= 1 && typeof headers[0] === "object") {
                headers = headers[0];
            }
        } catch (e) {
            headers = undefined;
        }

        let body = this.GetInputValue("body", cache);
        try {
            if(body && body.length >= 1 && typeof body[0] === "string") {
                if(body.length === 1) {
                    body = body[0];
                } else {
                    temp = {};
                    body.forEach(b => {
                        const parts = b.split(b.includes(": ") ? ": " : ":");
                        temp[parts[0]] = parts[1];
                    });
                    body = temp;
                }
            } else if(body && body.length >= 1 && typeof body[0] === "object") {
                body = body[0];
            }
        } catch (e) {
            body = undefined;
        }
        const method = this.GetOptionValue("method_type", cache);
        const return_type = this.GetOptionValue("return_type", cache);

        const axios = require("axios");
        axios({
            method: method,
            url: url,
            headers: headers ?? undefined,
            data: body ?? undefined,
            timeout: this.GetOptionValue("timeout", cache)
        }).then(response => {
            this.StoreOutputValue(response.status, "status", cache);
            this.StoreOutputValue(response.headers, "headers", cache);
            if(return_type === "json") {
                this.StoreOutputValue(response.data, "data", cache);
            } else if(return_type === "buffer") {
                this.StoreOutputValue(Buffer.from(response.data), "data", cache);
            } else if(return_type === "attachment") {
                this.StoreOutputValue(new AttachmentBuilder(Buffer.from(response.data)), "data", cache);
            }
            this.RunNextBlock("action", cache);
        }).catch(err => {
            console.log(err)
        })
    }
}